unit Cscprof;
{
  This unit provides a crude method to determine which sections of
  code are using the most execution time. To use :
    1) Add this Pascal file to your program.
    2) Add an enumerated type, or a list of integer constants starting
       at zero, that has entry for each block of code you want to profile.
    3) Call the pinit procedure with a list of names for these areas as
       a null terminated string, with the names separated by spaces.
    4) When you enter the block, call the penter procedure, passing the
       integer number of that block (defined in step 2).
    5) When you exit the block, call the pexit procedure, passing the
       integer number of that block (defined in step 2).
    5) Call the pdone procedure to terminate profiling.
    6) Call profreport.showmodal to display the results.
    7) Define DOPROF in your conditional defines list (PROJECT|DIRECTORIES)
       undefine DOPROF to run your code at full speed.

  Example :

  uses cscprof ;
type
  proflist = (P_BLOCK1, P_BLOCK2) ;           or P_BLOCK1 = 0 ; P_BLOCK2 = 1 ;
       .
       .
       .
  pinit ('BLOCK1 BLOCK2') ;
       .
       .
  penter (ord(P_BLOCK1)) ;                    or penter (P_BLOCK1) ;
       .
       .
  penter (ord(P_BLOCK2)) ;                    or penter (P_BLOCK2) ;
       .
       .
  pexit (ord(P_BLOCK2)) ;                     or pexit (P_BLOCK2) ;
       .
       .
  pexit (ord(P_BLOCK1)) ;                     or pexit (P_BLOCK1) ;
       .
       .
  pdone ;
       .
       .
  profreport.showmodal ;

  Note that you can nest blocks of code to be profiled, but you must exit
  the blocks in the reverse order of entering them. In the example above,
  the time spent in block2 (which is contained within block1) is deducted
  from the total time for block1. This is a very common condition when
  you profile procedures.
}
{
  Copyright issues :

    Copyright (C) 1996 reserved by Certified Software Corporation
                                   PO Box 70
                                   Randolph, VT 05060 USA
                                   certsoft@quest-net.com

    This unit is free software, you can redistribute it and/or
    modify it. This unit is distributed in the hope that it will be
    useful, but WITHOUT ANY WARRANTY, without even the implied warranty
    of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
}
interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  StdCtrls, ExtCtrls, Forms, Buttons, toolhelp;

type
  Tprofreport = class(TForm)
    BitBtn1: TBitBtn;
    proflist: TListBox;
    Label1: TLabel;
    Panel1: TPanel;
    totallab: TLabel;
    procedure FormShow(Sender: TObject);
  end;

var
  profreport: Tprofreport;

{$IFDEF DOPROF}
procedure pinit (names : pchar) ;
procedure penter (procnum : integer) ;
procedure pexit (procnum : integer) ;
procedure pdone ;
{$ENDIF}

implementation
const
  PROF_ENTRIES = 256 ; { maximum number of "blocks" to profile }
  STACK_SIZE = 100 ;   { maximum nesting of blocks at any one time }
type
  profile_type = array[0 .. PROF_ENTRIES - 1] of record
    entry_exit_time : longint ;
    timespent : longint ;
    invocations : longint ;
    inproc : integer ;
    procname : string[20]
  end ;

  procedure_stack_type = array [1 .. STACK_SIZE] of record
    calling_proc : integer ;
    entered_at : longint
  end ;

var
  profile : profile_type ;
  pstack : procedure_stack_type ;
  starttime : longint ;
  endtime : longint ;
  highproc : integer ;
  stacked : integer ;

{$R *.DFM}

{$IFDEF DOPROF}
function curms : longint ;
type
  ttimerinfo = record
    size : longint ;
    mssincestart : longint ;
    msthisvm : longint
  end ;
var
  timer : ttimerinfo ;
begin
  timer.size := sizeof(ttimerinfo) ;
  timercount(@timer) ;
  curms := timer.msthisvm
end ;

procedure pinit (names : pchar) ;
var
  i : integer ;
  s : string ;
begin
  highproc := -1 ;
  stacked := 0 ;
  i := 0 ;
  while (names[i] <> #0) and (highproc < PROF_ENTRIES) do
    begin
      while names[i] = ' ' do  {skip leading spaces}
        inc(i) ;
      s := '' ;
      while names[i] <> ' ' do {get name from list}
        begin
          if names[i] = #0
            then
              break ;
          appendstr(s, names[i]) ;
          inc(i)
        end ;
      if s <> ''
        then
          begin  { have a new name }
            inc(highproc) ;
            with profile[highproc] do
              begin
                timespent := 0 ;
                invocations := 0 ;
                inproc := 0 ;
                procname := copy(s, 1, 20)
              end
          end
    end ;
  starttime := curms
end ; { pinit }

procedure penter (procnum : integer) ;
var
  snap : longint ;
begin
  snap := curms ;
  if stacked < STACK_SIZE
    then
      begin
        inc(stacked) ;
        with pstack[stacked] do
          begin
            entered_at := snap ;
            calling_proc := procnum
          end ;
        with profile[procnum] do
          begin
            inproc := stacked ;
            inc(invocations) ;
            entry_exit_time := snap
          end
      end
end ;

procedure pexit (procnum : integer) ;
var
  snap, elapsed : longint ;
begin
  snap := curms ;
  with profile[procnum] do
    begin
      inproc := 0 ;
      elapsed := snap - entry_exit_time ;
      timespent := timespent + elapsed  { update time for this proc }
    end ;
  if stacked > 0
    then
      dec(stacked) ;
  if stacked > 0
    then
      with profile[pstack[stacked].calling_proc] do  { update enclosing proc }
        timespent := timespent - elapsed
end ;

procedure pdone ;
begin
  endtime := curms
end ;
{$ENDIF}

procedure Tprofreport.FormShow(Sender: TObject);
var
  proc : integer ;
  total : longint ;
  s : string ;
  s1 : string[40] ;
begin
  proflist.clear ;
  total := endtime - starttime ;
  for proc := 0 to highproc do
    with profile[proc] do
      begin
        s := procname ;
        while length(s) < 20 do
          appendstr(s, ' ') ;
        s := concat(s, inttostr(timespent)) ;
        while length(s) < 30 do
          appendstr(s, ' ') ;
        s := concat(s, inttostr(invocations)) ;
        while length(s) < 42 do
          appendstr(s, ' ') ;
        if invocations <> 0
          then
            begin
              str(timespent/invocations:4:4, s1) ;
              appendstr(s, s1) ;
              while length(s) < 51 do
                appendstr(s, ' ') ;
              str((timespent/total) * 100.0:4:2, s1) ;
              appendstr(s, s1)
            end ;
        proflist.items.add(s)
      end ;
  totallab.caption := 'Total Execution Time=' + inttostr(total) + 'MS' ;
end;

initialization
  highproc := -1 ;{ not setup }
  starttime := 0 ;
  endtime := 0 ;
end.
